// Usando delegación para elementos creados dinámicamente
$('body').on('focus', '.item-qty-input', function() {
    const $this = $(this);
    setTimeout(() => {
        $this.select();
    }, 0);
});

/**
 * Recorta una cadena de texto a una longitud máxima y añade puntos suspensivos.
 * @param {string} text - La cadena de texto a recortar (item.producto).
 * @param {number} maxLength - La longitud máxima permitida para la cadena.
 * @returns {string} El texto recortado o el texto original si es más corto.
 */
function recortarTexto(text, maxLength = 28) {
    if (!text) {
        return ''; // Devuelve una cadena vacía si no hay texto
    }
    // Si la longitud del texto es menor o igual a la longitud máxima, devuelve el texto completo.
    if (text.length <= maxLength) {
        return text;
    }
    // Si es más largo, recorta el texto y añade "...".
    return text.substring(0, maxLength) + '...';
}

// #region UTILS
const Utils = {
    /**
     * Formatea un número con separadores de miles y dos decimales.
     * @param {number|string} x - El número a formatear.
     * @returns {string} El número formateado.
     */
    Separador: function(x) { //SEPARADOR CON DECIMAL
        return parseFloat(x).toFixed(0).replace(/\B(?=(\d{3})+(?!\d))/g, ".");
    },
    /**
     * Permite solo la entrada de números y un punto decimal en un campo de texto.
     * @param {Event} event - El objeto de evento del teclado.
     * @returns {boolean} True si la tecla presionada es un número o un punto, false en caso contrario.
     */
    SoloNumeros: function(event) {
        const charCode = (event.which) ? event.which : event.keyCode;

        // Permite números (0-9)
        // Permite la tecla de retroceso (Backspace, charCode 8) para borrar
        // Permite la tecla de suprimir (Delete, charCode 46, pero aquí también es el punto)
        // Permite la tecla de tabulación (Tab, charCode 9) para navegar
        if (charCode >= 48 && charCode <= 57) { // Números 0-9
            return true;
        }

        // Permite el punto decimal si no hay uno ya en el campo
        if (charCode === 46) { // Tecla de punto (o suprimir en algunos casos)
            // Asegura que no se introduzcan múltiples puntos decimales
            if (event.target.value.indexOf('.') === -1) {
                return true;
            }
        }
        
        // Permite las teclas de navegación (flechas, inicio, fin)
        // charCode 37: Left arrow
        // charCode 39: Right arrow
        // charCode 8: Backspace
        // charCode 9: Tab
        // charCode 46: Delete (y el punto, ya manejado arriba)
        if (charCode === 8 || charCode === 9 || charCode === 37 || charCode === 39) {
            return true;
        }

        // Si no es un número ni un punto ni una tecla de control permitida
        return false;
    },
    /**
     * Limpia los campos de texto de entrada de productos después de agregar al carrito.
     */
    LimpiarTexto: function() {
        $("#idproducto").val("");
        $("#codproducto").val("");
        $("#producto").val("");
        $("#descripcion").val("");
        $("#opcionvendido").val("");
        $("#imei").val("");
        $("#condicion").val("");
        $("#codmarca").val("");
        $("#marcas").val("");
        $("#codmodelo").val("");
        $("#modelos").val("");
        $("#codpresentacion").val("");
        $("#presentacion").val("");
        $("#codcolor").val("");
        $("#color").val("");
        $("#preciocompra").val("");
        $("#precioventa").val("");
        $("#descproducto").val("0");
        $("#ivaproducto").val("");
        $("#existencia").val("0");
        $("#precioconiva").val("");
        $("#tipodetalle").val("1");
        $("#cantidad").val("1");
        $("#search_producto_barra").val("");
        //$("#search_producto").focus();
    },
    /**
     * Resetea los estilos de los campos de entrada de validación.
     */
    resetInputStyles: function() {
        $("#cantidad").css('border-color', '');
    },
    /**
     * Muestra un mensaje de alerta.
     * @param {string} message - El mensaje a mostrar.
     * @param {string} type - Tipo de alerta ('success', 'error', 'warning', 'info').
     * @param {string|null} inputId - ID del input para enfocar y resaltar (opcional).
     */
    showAlert: function(message, type = "error", inputId = null) {
        swal("Oops", message, type);
        if (inputId) {
            $(inputId).focus();
            $(inputId).css('border-color', '#ff7676');
        }
    }
};
// #endregion UTILS

// #region CALCULADORA DE CARRITO (Mejorada)
const _cartCalculator = {
    /**
     * Calcula los totales del carrito con redondeo a la cifra más cercana para números enteros.
     * @param {Array<Object>} items - Array de productos en el carrito.
     * @returns {Object} Un objeto con todos los totales calculados.
     */
    calculateTotals: function(items) {
        let totals = {
            OperacionItems: 0,
            TotalDescuentoIndividual: 0,
            SubtotalGeneral: 0,
            SubtotalExento: 0,
            BaseImpIva: 0,
            TotalIvaGeneral: 0,
            TotalCompra: 0,
            ValorExento: 0,
            SubtotalIva: 0,
            ValorSubtotalIva: 0,
            ValorTotalIva: 0,
            TotalDescuentoGeneral: 0,
            SubtotalConImpuesto: 0,
            SubtotalSinImpuesto: 0,
            ValorSubTotalFactura: 0,
            TotalFactura: 0,
            TotalFacturaCambio: 0,
            TipoCliente: $("#exonerado").val(),
            EditarPrecio: $("#editarprecio").val(),
            TipoDescuento: $("#tipodescuento").val()
        };

        if (!items || items.length === 0) {
            return totals;
        }

        let currentTotalCompra = 0;
        let currentTotalDescuentoIndividual = 0;
        let currentSubtotalGeneral = 0;
        let currentBaseImpIva = 0;
        let currentTotalIvaGeneral = 0;

        items.forEach(item => {
            if (parseFloat(item.cantidad) !== 0) {
                const cantsincero = parseFloat(item.cantidad);
                const PrecioVenta = parseFloat(item.precio2);

                currentTotalCompra += parseFloat(item.precio) * cantsincero;

                const DetalleDescuento = (PrecioVenta * cantsincero) * (parseFloat(item.descproducto) / 100);
                currentTotalDescuentoIndividual += DetalleDescuento;

                const descsiniva = PrecioVenta * parseFloat(item.descproducto) / 100;
                const Operacion = (PrecioVenta - descsiniva) * cantsincero;
                currentSubtotalGeneral += Operacion;

                if (item.ivaproducto !== "0" && parseFloat(item.ivaproducto) !== 0) {
                    const PrecioConIva = parseFloat(item.precioconiva);
                    const descconiva = PrecioConIva * parseFloat(item.descproducto) / 100;
                    const OperacionImpuestoCantidad = (PrecioConIva - descconiva) * cantsincero;
                    currentBaseImpIva += OperacionImpuestoCantidad;

                    const iva = parseFloat(item.ivaproducto) / 100;
                    const ivaFactor = 1 + iva;
                    const SubtotalDiscriminado = OperacionImpuestoCantidad / ivaFactor;
                    const BaseDiscriminado = OperacionImpuestoCantidad - SubtotalDiscriminado;
                    currentTotalIvaGeneral += BaseDiscriminado;
                }
            }
        });

        const PorcentajeDescuentoGeneral = parseFloat($('#descuento').val()) / 100 || 0;
        const SubtotalSinIva             = currentSubtotalGeneral - currentTotalIvaGeneral;
        const TotalFinalSinDescuento     = SubtotalSinIva + currentTotalIvaGeneral;
        const TotalDescuentoGeneralFinal = TotalFinalSinDescuento * PorcentajeDescuentoGeneral;
        
        // Usamos Math.round() para obtener el resultado deseado de 71
        const TotalFacturaFinal = Math.round(TotalFinalSinDescuento - TotalDescuentoGeneralFinal);
        
        const montocambio             = parseFloat($('input#montocambio').val()) || 1;
        const TotalFacturaCambioFinal = TotalFacturaFinal / montocambio;

        totals.OperacionItems           = Math.round(items.reduce((sum, item) => sum + (parseFloat(item.cantidad) || 0), 0));
        totals.TotalDescuentoIndividual = Math.round(currentTotalDescuentoIndividual);
        totals.SubtotalGeneral          = Math.round(currentSubtotalGeneral);
        totals.TotalCompra              = Math.round(currentTotalCompra);
        totals.TotalDescuentoGeneral    = Math.round(TotalDescuentoGeneralFinal);
        totals.ValorExento              = Math.round(currentSubtotalGeneral - currentBaseImpIva);
        totals.ValorSubtotalIva         = Math.round(currentBaseImpIva - currentTotalIvaGeneral);
        totals.ValorTotalIva            = Math.round(currentTotalIvaGeneral);
        totals.ValorSubTotalFactura     = Math.round(totals.ValorExento + totals.ValorSubtotalIva);
        totals.TotalFactura             = TotalFacturaFinal;
        totals.TotalFacturaCambio       = TotalFacturaCambioFinal;

        return totals;
    }
};
// #endregion CALCULADORA DE CARRITO (Mejorada)

// #region UI UPDATER
const _uiUpdater = {
    /**
     * Renderiza la tabla del carrito con los items y actualiza los resúmenes.
     * @param {Array<Object>} items - Array de objetos que representan los productos en el carrito.
     * @param {Object} totals - Objeto con los totales calculados por _cartCalculator.
     */
    updateCartUI: function(items, totals) {

        //console.log("-----------------------------------------");
        //console.log("DEBUG UI UPDATER: Totales recibidos para actualizar UI:", totals);
        //console.log("DEBUG UI UPDATER: Items recibidos para actualizar UI:", items);

        $("#carrito tbody").html("");
        if (items.length === 0) {
            this.resetSalesUI();
            return; // Importante para salir de la funció
        } else {
            items.forEach(item => {
                const cantsincero = parseFloat(item.cantidad);
                if (cantsincero !== 0) {
                    let Detalle = "";
                    if (item.tipodetalle === "1") {
                        Detalle = "<span class='badge badge-success font-12'>PRODUCTO</span>";
                    } else if (item.tipodetalle === "2") {
                        Detalle = "<span class='badge badge-info font-12'>COMBO</span>";
                    } else if (item.tipodetalle === "3") {
                        Detalle = "<span class='badge badge-primary font-12'>SERVICIO</span>";
                    }

                    const PrecioVenta      = parseFloat(item.precio2);
                    const ItemDescuento    = parseFloat(item.descproducto);
                    const ValorTotal       = PrecioVenta * cantsincero;
                    const DetalleDescuento = ValorTotal * parseFloat(item.descproducto) / 100;

                    const descsiniva  = PrecioVenta * parseFloat(item.descproducto) / 100;
                    const Operac      = PrecioVenta - descsiniva;
                    const Operacion   = Operac * cantsincero;

                    const nuevaFila = `
                        <tr class="cart-row" data-idx="${item.id}">
                            <td>
                                <div class="fw-semibold">${recortarTexto(item.producto.replace(/'/g, "&apos;"))}</div>
                                <div class="small text-muted">#${item.txtCodigo}</div>
                            </td>
                            <td class="text-center">
                                <div class="btn-group btn-group-sm" role="group">
                                    <button class="btn btn-info btn-sm" style="cursor:pointer;border-radius:5px 0px 0px 5px;" onclick="addItem('${item.id}', '${item.txtCodigo}', '-1', '${item.producto}', '${item.descripcion}', '${item.opcionvendido}', '${item.imei}', '${item.condicion}', '${item.codmarca}', '${item.marca}', '${item.codmodelo}', '${item.modelo}', '${item.codpresentacion}', '${item.presentacion}', '${item.codcolor}', '${item.color}', '${item.precio}', '${item.precio2}', '${item.descproducto}', '${item.ivaproducto}', '${item.existencia}', '${item.precioconiva}', '${item.tipodetalle}', '-' )"
                                    type='button'>-</button>
                                    <input class="text-center item-qty-input" 
                                    style='width:50px;font-size:14px;background:#F0F8FA;font-weight:bold;border: 2px solid #B7B7BD;box-shadow: inset 0px 0px 0px 0px rgba(0,0,0,0.3);'
                                    data-item-id='${item.id}'
                                    data-item-codigo='${item.txtCodigo}'
                                    data-item-producto='${item.producto.replace(/'/g, "&apos;")}'
                                    data-item-descripcion='${item.descripcion.replace(/'/g, "&apos;")}'
                                    data-item-opcionvendido='${item.opcionvendido}'
                                    data-item-imei='${item.imei}'
                                    data-item-condicion='${item.condicion}'
                                    data-item-codmarca='${item.codmarca}'
                                    data-item-marca='${item.marca.replace(/'/g, "&apos;")}'
                                    data-item-codmodelo='${item.codmodelo}'
                                    data-item-modelo='${item.modelo.replace(/'/g, "&apos;")}'
                                    data-item-codpresentacion='${item.codpresentacion}'
                                    data-item-presentacion='${item.presentacion.replace(/'/g, "&apos;")}'
                                    data-item-codcolor='${item.codcolor}'
                                    data-item-color='${item.color.replace(/'/g, "&apos;")}'
                                    data-item-precio='${item.precio}'
                                    data-item-precio2='${item.precio2}'
                                    data-item-descproducto='${item.descproducto}'
                                    data-item-ivaproducto='${item.ivaproducto}'
                                    data-item-existencia='${item.existencia}'
                                    data-item-precioconiva='${item.precioconiva}'
                                    data-item-tipodetalle='${item.tipodetalle}'
                                    value='${item.cantidad}'
                                    style='width:40px;height:30px;'
                                    onchange='handleQuantityChange(this)'
                                    onkeypress="return Utils.SoloNumeros(event)">
                                    <button class="btn btn-info btn-sm" style="cursor:pointer;border-radius:0px 5px 5px 0px;" onclick="addItem('${item.id}', '${item.txtCodigo}', '+1', '${item.producto}', '${item.descripcion}', '${item.opcionvendido}', '${item.opcionimei}', '${item.condicion}', '${item.codmarca}', '${item.marca}', '${item.codmodelo}', '${item.modelo}', '${item.codpresentacion}', '${item.presentacion}', '${item.codcolor}', '${item.color}', '${item.precio}', '${item.precio2}', '${item.descproducto}', '${item.ivaproducto}', '${item.existencia}', '${item.precioconiva}', '${item.tipodetalle}', '+' )"
                                    type='button'>+</button>
                                </div>
                              </td>
                            <td class="text-end">${parseInt(totals.EditarPrecio) === 1 ?
                                `<input type='text' class='bold item-qty-input'
                                data-item-id='${item.id}'
                                data-item-codigo='${item.txtCodigo}'
                                data-item-cantidad='${item.cantidad}'
                                data-item-producto='${item.producto.replace(/'/g, "&apos;")}'
                                data-item-descripcion='${item.descripcion.replace(/'/g, "&apos;")}'
                                data-item-opcionvendido='${item.opcionvendido}'
                                data-item-imei='${item.imei}'
                                data-item-condicion='${item.condicion}'
                                data-item-codmarca='${item.codmarca}'
                                data-item-marca='${item.marca.replace(/'/g, "&apos;")}'
                                data-item-codmodelo='${item.codmodelo}'
                                data-item-modelo='${item.modelo.replace(/'/g, "&apos;")}'
                                data-item-codpresentacion='${item.codpresentacion}'
                                data-item-presentacion='${item.presentacion.replace(/'/g, "&apos;")}'
                                data-item-codcolor='${item.codcolor}'
                                data-item-color='${item.color.replace(/'/g, "&apos;")}'
                                data-item-precio='${item.precio}'
                                data-item-descproducto='${item.descproducto}'
                                data-item-ivaproducto='${item.ivaproducto}'
                                data-item-existencia='${item.existencia}'
                                data-item-precioconiva='${item.precioconiva}'
                                data-item-tipodetalle='${item.tipodetalle}'
                                value='${PrecioVenta.toFixed(0)}'
                                style='width:70px;height:30px;font-size:14px;background:#F0F8FA;font-weight:bold;border: 2px solid #B7B7BD;box-shadow: inset 0px 0px 0px 0px rgba(0,0,0,0.3);'
                                onchange='handlePriceChange(this)'>`
                                : `<strong>${PrecioVenta.toFixed(0)}</strong>`
                            }</td>
                            <td class="text-end">${Utils.Separador(Operacion)}</td>
                            <td class="text-center">
                            <span class="btn btn-link text-danger p-0" onclick="addItem('${item.id}', '${item.txtCodigo}', '0', '${item.producto}', '${item.descripcion}', '${item.opcionvendido}', '${item.imei}', '${item.condicion}', '${item.codmarca}', '${item.marca}', '${item.codmodelo}', '${item.modelo}', '${item.codpresentacion}', '${item.presentacion}', '${item.codcolor}', '${item.color}', '${item.precio}', '${item.precio2}', '${item.descproducto}', '${item.ivaproducto}', '${item.existencia}', '${item.precioconiva}', '${item.tipodetalle}', '=' )"
                                title="Eliminar"><svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="feather feather-trash-2 icon"><polyline points="3 6 5 6 21 6"></polyline><path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"></path><line x1="10" y1="11" x2="10" y2="17"></line><line x1="14" y1="11" x2="14" y2="17"></line></svg></span></td>
                        </tr>
                    `;
                    $(nuevaFila).appendTo("#carrito tbody");
                }
            });

            // Actualizar etiquetas de resumen
            $("#lblitems").text(totals.OperacionItems.toFixed(2));
            $("#lblsubtotal").text(Utils.Separador(totals.ValorSubTotalFactura));
            $("#lblexonerado").text(Utils.Separador((totals.TipoCliente === "2" ? totals.ValorSubTotalFactura : 0)));
            $("#lblexento").text(Utils.Separador((totals.TipoCliente !== "2" ? totals.ValorExento : 0)));
            $("#lblsubtotaliva").text(Utils.Separador((totals.TipoCliente !== "2" ? totals.ValorSubtotalIva : 0)));
            $("#lbliva").text(Utils.Separador((totals.TipoCliente !== "2" ? totals.ValorTotalIva : 0)));
            $("#lbldescontado").text(Utils.Separador(totals.TotalDescuentoIndividual));
            $("#lbldescuento").text(Utils.Separador(totals.TotalDescuentoGeneral));
            $("#lbltotal").text(Utils.Separador(totals.TotalFactura));

            // Actualizar campos de texto ocultos (para envío al servidor)
            $("#txtsubtotal").val(totals.ValorSubTotalFactura.toFixed(0));
            $("#txtsubtotal2").val(totals.SubtotalGeneral.toFixed(0));
            $("#txtexonerado").val((totals.TipoCliente === "2" ? totals.ValorSubTotalFactura.toFixed(0) : "0"));
            $("#txtexonerado2").val((totals.TipoCliente === "2" ? totals.SubtotalConImpuesto.toFixed(0) : "0")); // Considerar si debe ser SubtotalFactura o SubtotalConImpuesto
            $("#txtexento").val(totals.ValorExento.toFixed(0));
            $("#txtexento2").val(totals.SubtotalExento.toFixed(0));
            $("#txtsubtotaliva").val(totals.ValorSubtotalIva.toFixed(0));
            $("#txtsubtotaliva2").val(totals.SubtotalIva.toFixed(0));
            $("#txtIva").val(totals.ValorTotalIva.toFixed(0));
            $("#txtIva2").val(totals.TotalIvaGeneral.toFixed(0));
            $("#txtdescontado").val(totals.TotalDescuentoIndividual.toFixed(0));
            $("#txtDescuento").val(totals.TotalDescuentoGeneral.toFixed(0));
            $("#txtTotal").val(totals.TotalFactura.toFixed(0));
            $("#txtPagado").val(totals.TotalFactura.toFixed(0));
            $("#txtTotalCompra").val(totals.TotalCompra.toFixed(0));

            // Activar botones de pago y guardar
            $("#buttonpago").attr('disabled', false);
            $("#submit_guardar").attr('disabled', false);
            $("#TextImporte").text(Utils.Separador(totals.TotalFactura));
            $("#TextPagado").text(Utils.Separador(totals.TotalFactura));
            $("#montopagado").val(totals.TotalFactura.toFixed(0));
        }
        $("#muestra_foto").html("<img src='fotos/ninguna.png' width='160' height='170'>");
        Utils.LimpiarTexto();
    },

    /**
     * Limpia completamente la interfaz de usuario de ventas.
     */
    resetSalesUI: function() {
        //$("#carrito tbody").html("<tr></tr>");
        $("#carrito tbody").html("<tr class='warning-element' style='border-left: 2px solid #ff5050 !important; background: #fce3e3;'><td class='text-center' colspan=5><h6>NO HAY DETALLES AGREGADOS</h6></td></tr>");
        $("#muestra_foto").html("<img src='fotos/ninguna.png' width='160' height='170'>");
        $("#lblitems, #lblsubtotal, #lblexento, #lblsubtotaliva, #lbliva, #lbldescontado, #lbldescuento, #lbltotal").text("0");
        $("#txtsubtotal, #txtsubtotal2, #txtexonerado, #txtexonerado2, #txtexento, #txtexento2, #txtsubtotaliva, #txtsubtotaliva2, #txtIva, #txtIva2, #txtdescontado, #descuento, #txtDescuento, #txtTotal, #txtPagado, #txtTotalCompra").val("0");
        $("#buttonpago").attr('disabled', true);
        $("#submit_guardar").attr('disabled', true);
        $("#TextImporte").text("0");
        $("#TextPagado").text("0");
        $("#TextCambio").text("0");
        $("#montopagado").val("0");
        $("#muestra_condiciones").load("condiciones_pagos.php?BuscaCondicionesPagos=si&tipopago=CONTADO&txtTotal=0.00");
        Utils.LimpiarTexto();
    }
};
// #endregion UI UPDATER

// #region CART MANAGER
const _cartManager = {
    /**
     * Envia una solicitud AJAX para manipular el carrito en el servidor.
     * @param {Object} cartItem - Objeto con los datos del producto/servicio y la operación.
     */
    updateCart: function(cartItem) {
        const DatosJson = JSON.stringify(cartItem);

        $.ajax({ // Cambiamos de $.post a $.ajax para un control más preciso
            url: "carritoventa.php",
            method: "POST", // Especificamos el método
            data: { MiCarrito: DatosJson },
            dataType: "json", // ¡Esto es CRUCIAL para asegurar que 'data' sea un JSON parseado!
            success: function(data) {
                //console.log("CART MANAGER: Datos recibidos de carritoventa.php (SUCCESS AJAX):", data);
                if (!data || data.length === 0) { // Si la respuesta es un array vacío
                    //console.log("DEBUG: _cartManager detectó carrito vacío desde PHP. Forzando UI a cero.");
                    _uiUpdater.resetSalesUI(); // Llama a la función que pone los campos de totales a "0.00"
                    return; // Terminamos aquí
                }
                // Si 'data' NO está vacío, procedemos con el cálculo y la actualización normal
                const totals = _cartCalculator.calculateTotals(data);
                //console.log("CART MANAGER: Totales calculados (SUCCESS AJAX):", totals);
                _uiUpdater.updateCartUI(data, totals);
                //console.log("CART MANAGER: _uiUpdater.updateCartUI debería haber actualizado los totales.");
            },
            error: function(jqXHR, textStatus, errorThrown) {
                //console.error("Error al actualizar el carrito:", textStatus, errorThrown);
                Utils.showAlert("Ha Ocurrido un Error al comunicarse con el servidor de carrito!", "error");
            }
        });
    },

    /**
     * Vacia el carrito de ventas (nuevo) en el servidor y actualiza la UI.
     */
    clearCart: function() {
        const Carrito = {
            Id: "vaciar",
            Codigo: "vaciar",
            Producto: "vaciar",
            Descripcion: "vaciar",
            OpcionVendido: "vaciar",
            Imei: "vaciar",
            Condicion: "vaciar",
            Codmarca: "vaciar",
            Marca: "vaciar",
            Codmodelo: "vaciar",
            Modelo: "vaciar",
            Codpresentacion: "vaciar",
            Presentacion: "vaciar",
            Codcolor: "vaciar",
            Color: "vaciar",
            Precio: "0",
            Precio2: "0",
            Descproducto: "0",
            Ivaproducto: "vaciar",
            Existencia: "vaciar",
            Precioconiva: "0",
            TipoDetalle: "vaciar",
            Cantidad: "0"
        };
        const DatosJson = JSON.stringify(Carrito);
        $.ajax({ // Cambiamos a $.ajax aquí también
            url: "carritoventa.php",
            method: "POST",
            data: { MiCarrito: DatosJson },
            dataType: "json",
            success: function(data) {
                //console.log("CART MANAGER: Respuesta de vaciar carrito (SUCCESS AJAX):", data);
                if (!data || data.length === 0) {
                    $("#saveposopcion3")[0].reset();
                    $("#codcliente").val("0");
                    $("#nrodocumento").val("0");
                    $("#exonerado").val("0");
                    $('#TextCliente').text("Consumidor Final");
                    $('#TextCredito').text("0");
                    $("#buttonpago").attr('disabled', true);
                    $("#submit_guardar").attr('disabled', true);
                    _uiUpdater.resetSalesUI(); // Reinicia la UI
                    //console.log("DEBUG: Carrito vaciado y UI reseteada.");
                } else {
                    const totals = _cartCalculator.calculateTotals(data);
                    _uiUpdater.updateCartUI(data, totals);
                }
            },
            error: function(jqXHR, textStatus, errorThrown) {
                //console.error("Error al vaciar el carrito:", textStatus, errorThrown);
                Utils.showAlert("Ha Ocurrido un Error al comunicarse con el servidor para vaciar el carrito!", "error");
            }
        });
    }
};
// #endregion CART MANAGER

// #region CORE FUNCTIONS (Manteniendo la API original)
/**
* Agrega o modifica un item en el carrito. Esta es la función principal que se llama desde los botones y la tabla.
*/
function addItem(idproducto, codproducto, cantidad, producto, descripcion, opcionvendido, imei, condicion, codmarca, marca, codmodelo, modelo, codpresentacion, presentacion, codcolor, color, preciocompra, precioventa, descproducto, ivaproducto, existencia, precioconiva, tipodetalle, opCantidad) { //

    const Carrito = {
        Id: idproducto,
        Codigo: codproducto,
        Producto: producto,
        Descripcion: descripcion,
        OpcionVendido: opcionvendido,
        Imei: imei,
        Condicion: condicion,
        Codmarca: codmarca,
        Marca: marca,
        Codmodelo: codmodelo,
        Modelo: modelo,
        Codpresentacion: codpresentacion,
        Presentacion: presentacion,
        Codcolor: codcolor,
        Color: color,
        Precio: preciocompra,
        Precio2: precioventa,
        Descproducto: descproducto,
        Ivaproducto: ivaproducto,
        Existencia: existencia,
        Precioconiva: precioconiva,
        TipoDetalle: tipodetalle,
        Cantidad: cantidad,
        opCantidad: opCantidad
    };
    _cartManager.updateCart(Carrito);
}

/**
* Realiza la acción de añadir un producto con cantidad 1 al carrito.
*/
function DoAction(idproducto, codproducto, producto, descripcion, opcionvendido, imei, condicion, codmarca, marca, codmodelo, modelo, codpresentacion, presentacion, codcolor, color, preciocompra, precioventa, descproducto, ivaproducto, existencia, precioconiva, tipodetalle) { //
    addItem(idproducto, codproducto, 1, producto, descripcion, opcionvendido, imei, condicion, codmarca, marca, codmodelo, modelo, codpresentacion, presentacion, codcolor, color, preciocompra, precioventa, descproducto, ivaproducto, existencia, precioconiva, tipodetalle, '+=');
}


/**
 * Maneja el cambio de cantidad en el input de una fila del carrito.
 * @param {HTMLInputElement} inputElement - El elemento input que disparó el evento.
 */
window.handleQuantityChange = function(inputElement) {
    const newQuantity = parseFloat(inputElement.value);

    // Si la cantidad es 0, o un número negativo, o no es un número, manejamos el error.
    // Opcionalmente, puedes permitir 0 para eliminar el ítem.
    if (isNaN(newQuantity) || newQuantity < 0) {
        // Revertir el valor del input a la última cantidad válida conocida o a 1
        inputElement.value = inputElement.defaultValue || 1; // Usar defaultValue o un valor por defecto
        // swal("Advertencia", "La cantidad debe ser un número válido mayor o igual a cero.", "warning");
        return;
    }

    // Opcional: Si quieres que el usuario no pueda teclear 0 para eliminar,
    // puedes forzarlo a 1 si teclea 0 y el producto no es de IMEI (que permite 0 temporalmente)
    if (newQuantity === 0 && inputElement.dataset.itemOpcionimei !== "SI") {
         // Si es un producto normal (no IMEI) y teclea 0, puedes forzar a 1
         inputElement.value = 1;
         // swal("Advertencia", "La cantidad mínima para este producto es 1. Para eliminarlo, usa el botón de eliminar.", "warning");
         // return; // Si quieres que no haga la llamada a addItem
    }


    // Extraer todas las propiedades del ítem de los atributos data-
    // Asegúrate de que los nombres de las propiedades coincidan con los que espera addItem
    const item = {
        id: inputElement.dataset.itemId,
        txtCodigo: inputElement.dataset.itemCodigo,
        producto: inputElement.dataset.itemProducto.replace(/&apos;/g, "'"), // Deshacer el escape de comillas simples
        descripcion: inputElement.dataset.itemDescripcion.replace(/&apos;/g, "'"),
        opcionvendido: inputElement.dataset.itemOpcionVendido,
        imei: inputElement.dataset.itemimei,
        condicion: inputElement.dataset.itemCondicion,
        codmarca: inputElement.dataset.itemCodmarca,
        marca: inputElement.dataset.itemMarca.replace(/&apos;/g, "'"),
        codmodelo: inputElement.dataset.itemCodmodelo,
        modelo: inputElement.dataset.itemModelo.replace(/&apos;/g, "'"),
        codpresentacion: inputElement.dataset.itemCodpresentacion,
        presentacion: inputElement.dataset.itemPresentacion.replace(/&apos;/g, "'"),
        codcolor: inputElement.dataset.itemCodcolor,
        color: inputElement.dataset.itemColor.replace(/&apos;/g, "'"),
        precio: inputElement.dataset.itemPrecio,
        precio2: inputElement.dataset.itemPrecio2,
        descproducto: inputElement.dataset.itemDescproducto,
        ivaproducto: inputElement.dataset.itemIvaproducto,
        existencia: inputElement.dataset.itemExistencia,
        precioconiva: inputElement.dataset.itemPrecioconiva,
        tipodetalle: inputElement.dataset.itemTipodetalle
    };

    // Actualizar el valor predeterminado del input para futuras comparaciones
    // (Útil si no se re-renderiza todo el carrito inmediatamente después del cambio)
    inputElement.defaultValue = newQuantity.toString();


    // Llamar a addItem para actualizar el carrito con la nueva cantidad absoluta
    // El segundo argumento 'cantidad' de addItem será la nueva cantidad.
    // El último argumento 'opCantidad' será '=' para indicar que se debe establecer la cantidad.
    addItem(
        item.id,
        item.txtCodigo,
        newQuantity.toString(), // Pasa la nueva cantidad como el argumento 'cantidad'
        item.producto,
        item.descripcion,
        item.opcionvendido,
        item.imei,
        item.condicion,
        item.codmarca,
        item.marca,
        item.codmodelo,
        item.modelo,
        item.codpresentacion,
        item.presentacion,
        item.codcolor,
        item.color,
        item.precio,
        item.precio2,
        item.descproducto,
        item.ivaproducto,
        item.existencia,
        item.precioconiva,
        item.tipodetalle,
        '=' // Indica al backend que la operación es "establecer esta cantidad"
    );
};



/**
 * Maneja el cambio de Precio Venta en el input de una fila del carrito.
 * @param {HTMLInputElement} inputElement - El elemento input que disparó el evento.
 */
window.handlePriceChange = function(inputElement) {
    const newPrice = parseFloat(inputElement.value);

    // Extraer todas las propiedades del ítem de los atributos data-
    // Asegúrate de que los nombres de las propiedades coincidan con los que espera addItem
    const item = {
        id: inputElement.dataset.itemId,
        txtCodigo: inputElement.dataset.itemCodigo,
        cantidad: inputElement.dataset.itemCantidad,
        producto: inputElement.dataset.itemProducto.replace(/&apos;/g, "'"), // Deshacer el escape de comillas simples
        descripcion: inputElement.dataset.itemDescripcion.replace(/&apos;/g, "'"),
        opcionvendido: inputElement.dataset.itemOpcionVendido,
        imei: inputElement.dataset.itemimei,
        condicion: inputElement.dataset.itemCondicion,
        codmarca: inputElement.dataset.itemCodmarca,
        marca: inputElement.dataset.itemMarca.replace(/&apos;/g, "'"),
        codmodelo: inputElement.dataset.itemCodmodelo,
        modelo: inputElement.dataset.itemModelo.replace(/&apos;/g, "'"),
        codpresentacion: inputElement.dataset.itemCodpresentacion,
        presentacion: inputElement.dataset.itemPresentacion.replace(/&apos;/g, "'"),
        codcolor: inputElement.dataset.itemCodcolor,
        color: inputElement.dataset.itemColor.replace(/&apos;/g, "'"),
        precio: inputElement.dataset.itemPrecio,
        descproducto: inputElement.dataset.itemDescproducto,
        posicionimpuesto: inputElement.dataset.itemPosicionimpuesto,
        tipoimpuesto: inputElement.dataset.itemTipoimpuesto,
        ivaproducto: inputElement.dataset.itemIvaproducto,
        existencia: inputElement.dataset.itemExistencia,
        precioconiva: inputElement.dataset.itemPrecioconiva,
        tipodetalle: inputElement.dataset.itemTipodetalle
    };

    // Actualizar el valor predeterminado del input para futuras comparaciones
    // (Útil si no se re-renderiza todo el carrito inmediatamente después del cambio)
    inputElement.defaultValue = newPrice.toString();

    // Llamar a addItem para actualizar el carrito con la nueva cantidad absoluta
    // El último argumento 'opCantidad' será '=' para indicar que se debe establecer la cantidad.
    addItem(
        item.id,
        item.txtCodigo,
        item.cantidad,
        item.producto,
        item.descripcion,
        item.opcionvendido,
        item.imei,
        item.condicion,
        item.codmarca,
        item.marca,
        item.codmodelo,
        item.modelo,
        item.codpresentacion,
        item.presentacion,
        item.codcolor,
        item.color,
        item.precio,
        newPrice.toString(),
        item.descproducto,
        item.ivaproducto,
        item.existencia,
        item.precioconiva,
        item.tipodetalle,
        '='
    );
};

/**
 * Maneja el cambio de Descuento de Producto en el input de una fila del carrito.
 * @param {HTMLInputElement} inputElement - El elemento input que disparó el evento.
 */
window.handleDescuentoChange = function(inputElement,inputElement2) {
    const newDescuento = parseFloat(inputElement.value);
    const MontoLimite  = parseFloat($('#limite_descuento').val()) || 0;

    if (newDescuento > MontoLimite) {// Primer chequeo: si el descuento sobrepasa el límite permitido
        inputElement.value = inputElement2.toFixed(0);
        Utils.showAlert("NO TIENE AUTORIZACIÓN PARA ASIGNAR ESE PORCENTAJE EN DESCUENTO, SU LIMITE MAXIMO ES " + $('input#limite_descuento').val() + "%, COMUNIQUESE CON EL ADMINISTRADOR PARA SUBIR EL LIMITE!", "error");
        return false;
    } else if (newDescuento > 100) {// Segundo chequeo: si el descuento sobrepasa el 100%
        inputElement.value = inputElement2.toFixed(0);
        Utils.showAlert("EL DESCUENTO NO PUEDE SER MAYOR A 100% !", "error");
        return false;
    }

    // Extraer todas las propiedades del ítem de los atributos data-
    const item = {
        id: inputElement.dataset.itemId,
        txtCodigo: inputElement.dataset.itemCodigo,
        cantidad: inputElement.dataset.itemCantidad,
        producto: inputElement.dataset.itemProducto.replace(/&apos;/g, "'"), // Deshacer el escape de comillas simples
        descripcion: inputElement.dataset.itemDescripcion.replace(/&apos;/g, "'"),
        opcionvendido: inputElement.dataset.itemOpcionVendido,
        imei: inputElement.dataset.itemimei,
        condicion: inputElement.dataset.itemCondicion,
        codmarca: inputElement.dataset.itemCodmarca,
        marca: inputElement.dataset.itemMarca.replace(/&apos;/g, "'"),
        codmodelo: inputElement.dataset.itemCodmodelo,
        modelo: inputElement.dataset.itemModelo.replace(/&apos;/g, "'"),
        codpresentacion: inputElement.dataset.itemCodpresentacion,
        presentacion: inputElement.dataset.itemPresentacion.replace(/&apos;/g, "'"),
        codcolor: inputElement.dataset.itemCodcolor,
        color: inputElement.dataset.itemColor.replace(/&apos;/g, "'"),
        precio: inputElement.dataset.itemPrecio,
        precio2: inputElement.dataset.itemPrecio2,
        ivaproducto: inputElement.dataset.itemIvaproducto,
        existencia: inputElement.dataset.itemExistencia,
        precioconiva: inputElement.dataset.itemPrecioconiva,
        tipodetalle: inputElement.dataset.itemTipodetalle
    };

    // Actualizar el valor predeterminado del input para futuras comparaciones
    // (Útil si no se re-renderiza todo el carrito inmediatamente después del cambio)
    inputElement.defaultValue = newDescuento.toString();

    // Llamar a addItem para actualizar el carrito con la nueva cantidad absoluta
    // El último argumento 'opCantidad' será '=' para indicar que se debe establecer la cantidad.
    addItem(
        item.id,
        item.txtCodigo,
        item.cantidad,
        item.producto,
        item.descripcion,
        item.opcionvendido,
        item.imei,
        item.condicion,
        item.codmarca,
        item.marca,
        item.codmodelo,
        item.modelo,
        item.codpresentacion,
        item.presentacion,
        item.codcolor,
        item.color,
        item.precio,
        item.precio2,
        newDescuento.toString(),
        item.ivaproducto,
        item.existencia,
        item.precioconiva,
        item.tipodetalle,
        '='
    );
};

/**
 * Función que se ejecuta al seleccionar un cliente.
 * Verifica si el cliente es exonerado y recalcula el carrito si es necesario.
 */
function EjecutarFuncion() {
    // Obtenemos el valor actual del campo 'exonerado'
    const estadoExonerado = $("#exonerado").val(); // Este valor puede ser '0', '1', o '2'

    // Llama a updateCart con un objeto vacío para que recargue el carrito desde la sesión
    // y aplique los cálculos de totales basándose en el nuevo estado del cliente (exonerado/no exonerado).
    // Esto es crucial para que _cartManager.updateCart a su vez llame a _cartCalculator.calculateTotals
    // con el valor actualizado de #exonerado.
    console.log(`DEBUG: El campo #exonerado tiene valor '${estadoExonerado}'. Recalculando carrito.`);
    _cartManager.updateCart({});
}
// #endregion CORE FUNCTIONS

// #region EVENT HANDLERS
$(document).ready(function() {
    /*############ FUNCION DESACTIVA ENTER EN FORMULARIO ############*/
    $('#saveposopcion3').keypress(function(e) { //
        const keycode = (e.keyCode ? e.keyCode : e.which);
        if (keycode === 13) {
            return false;
        }
    });

    /*############ FUNCION AGREGA POR BOTON ############*/
    $('#AgregaProducto').click(function() { //
        AgregaDetalle();
    });

    /*############ FUNCION AGREGA POR CRITERIO ############*/
    $('#cantidad').keypress(function(e) { //
        const keycode = (e.keyCode ? e.keyCode : e.which);
        if (keycode === 13) {
            AgregaDetalle();
            e.preventDefault();
            return false;
        }
    });

    /* #################### FUNCION UNIFICADA DE BUSQUEDA #################### */
    $(function() {
        // Lógica para Busqueda de Barra
        function BuscarYAgregarProducto(codigo, cantidad) {
            let Impuesto = $('input#iva').val();
            $.ajax({
                url: "class/busqueda_autocompleto.php?Busqueda_Producto_Barcode=si",
                data: {
                    barcode: codigo
                },
                type: 'POST',
                dataType: 'json',
                success: function(json) {
                    // Si el servidor no devuelve un producto válido...
                    if (!json || !json[0] || !json[0].idproducto) {
                        // Muestra una alerta, limpia el campo y termina la función
                        Utils.showAlert("PRODUCTO NO ENCONTRADO O DATOS INVÁLIDOS.", "warning", "#search_producto_barra");
                        $("#search_producto_barra").val("").focus();
                        return false;
                    }

                    // Si todo está bien, crea el objeto del producto y lo agrega al carrito
                    const itemToAdd = {
                        Id: json[0].idproducto,
                        Codigo: json[0].txtCodigo,
                        Producto: json[0].producto,
                        Descripcion: (json[0].descripcion === "" ? "0" : json[0].descripcion),
                        OpcionVendido: json[0].opcionvendido,
                        Imei: json[0].imei,
                        Condicion: (json[0].condicion === "" ? "0" : json[0].condicion),
                        Codmarca: json[0].codmarca,
                        Marca: (json[0].codmarca === "0" ? "*****" : json[0].nommarca),
                        Codmodelo: json[0].codmodelo,
                        Modelo: (json[0].codmodelo === "0" ? "*****" : json[0].nommodelo),
                        Codpresentacion: json[0].codpresentacion,
                        Presentacion: (json[0].codpresentacion === "0" ? "******" : json[0].nompresentacion),
                        Codcolor: json[0].codcolor,
                        Color: (json[0].codcolor === "0" ? "******" : json[0].nomcolor),
                        Precio: json[0].preciocompra,
                        Precio2: json[0].precioxpublico,
                        Descproducto: json[0].descproducto,
                        Ivaproducto: (json[0].ivaproducto === "NO" ? "0" : Impuesto),
                        Existencia: json[0].existencia,
                        Precioconiva: (json[0].ivaproducto === "0" ? "0" : json[0].precioxpublico),
                        TipoDetalle: "1",
                        Cantidad: cantidad, // <-- ¡Aquí se usa la cantidad que detectamos!
                        opCantidad: '+='
                    };
                    _cartManager.updateCart(itemToAdd);
                    $("#search_producto_barra").val("").focus();
                },
                error: function(xhr, status, error) {
                    // Muestra el error exacto en la Consola del navegador (muy útil para nosotros)
                    console.error("Error en la llamada AJAX:", status, error, xhr); 

                    // Muestra una alerta simple para el usuario
                    Utils.showAlert("Error en la comunicación con el servidor. Revisa tu conexión.", "error");
                    $("#search_producto_barra").val("").focus();
                }
            });
        }
        
        // Lógica para autocompletado
        $("#search_producto_barra").autocomplete({
            source: function(request, response) {
                if (request.term.length >= 6 && !isNaN(request.term)) {
                    response([]);
                    return;
                }
                $.ajax({
                    url: "class/busqueda_autocompleto.php?Busqueda_Productos=si",
                    data: { term: request.term },
                    dataType: "json",
                    success: function(data) { response(data); }
                });
            },
            minLength: 1,
            select: function(event, ui) {
                if (ui.item) {
                    // Llama directamente a la función unificada con cantidad 1
                    BuscarYAgregarProducto(ui.item.codproducto, 1);
                    $("#search_producto_barra").val("").focus();
                    return false;
                }
            }
        });

        // función de Verificador Despachador
        $('#search_producto_barra').on('keypress', function(e) {
            if (e.which === 13) {
                e.preventDefault();

                const codeBar = $(this).val().trim();
                $(this).autocomplete("close");

                // Condición 1: Lógica para formato `cantidad*código`
                if (/^\d*([.,]\d+)?\*[\w\-]+$/.test(codeBar)) {
                    let [cantidadStr, codproducto] = codeBar.split('*');
                    let cantidad = parseFloat(cantidadStr.replace(',', '.'));
                    BuscarYAgregarProducto(codproducto, cantidad);

                // Condición 5: Código de barras normal
                } else {
                    if (codeBar.length > 0) {
                        const codigoNormal = codeBar.replace(/^0+/, '');
                        BuscarYAgregarProducto(codigoNormal, 1);
                    } else {
                        // Si el campo está vacío, simplemente regresamos el foco y no hacemos nada.
                        $("#search_producto_barra").val("").focus();
                    }
                }
            }
        }); 
    });
    /* #################### FUNCION UNIFICADA DE BUSQUEDA #################### */

    /* CANCELAR LOS ITEM AGREGADOS EN REGISTRO */
    $("#vaciar").click(function() { //
        _cartManager.clearCart();
    });

    // Cargar el carrito al iniciar la página
    _cartManager.updateCart({}); // Enviar un objeto vacío para que PHP devuelva el carrito actual.

    // La forma más robusta es monitorear el cambio en el input hidden 'exonerado'.
    // Esto asegura que la función se dispare cuando el valor de 'exonerado' se establece
    // (ya sea por una función de autocompletado, un modal, o cualquier otro script que actualice el campo).
    $('#exonerado').on('change', function() {
        console.log("DEBUG: El campo #exonerado ha cambiado. Llamando a EjecutarFuncion().");
        EjecutarFuncion();
    });

    // FUNCION PARA ACTUALIZAR DESCUENTO EN DETALLE
    $(document).on('keyup', '#descproducto', function() {
        const MontoDescuento = parseFloat($('#descproducto').val()) || 0;
        const MontoLimite    = parseFloat($('#limite_descuento').val()) || 0;
        if (MontoDescuento > MontoLimite) {// Primer chequeo: si el descuento sobrepasa el límite permitido
            $("#descproducto").val("0");
            Utils.showAlert("NO TIENE AUTORIZACIÓN PARA ASIGNAR ESE PORCENTAJE EN DESCUENTO, SU LIMITE MAXIMO ES " + $('input#limite_descuento').val() + "%, COMUNIQUESE CON EL ADMINISTRADOR PARA SUBIR EL LIMITE!", "error");
            return false;
        } else if (MontoDescuento > 100) {// Segundo chequeo: si el descuento sobrepasa el 100%
            $("#descproducto").val("0");
            Utils.showAlert("EL DESCUENTO NO PUEDE SER MAYOR A 100% !", "error");
            return false;
        }
    });

    // FUNCION PARA ACTUALIZAR CALCULO EN FACTURA DE COMPRAS CON DESCUENTO
    $(document).on('keyup', '#descuento', function() {
        const MontoDescuento = parseFloat($(this).val()) || 0;
        const MontoLimite = parseFloat($('#limite_descuento').val()) || 0;
        if (MontoDescuento > MontoLimite) {// Primer chequeo: si el descuento sobrepasa el límite permitido
            $(this).val("0");
            Utils.showAlert("NO TIENE AUTORIZACIÓN PARA ASIGNAR ESE PORCENTAJE EN DESCUENTO, SU LIMITE MAXIMO ES " + $('input#limite_descuento').val() + "%, COMUNIQUESE CON EL ADMINISTRADOR PARA SUBIR EL LIMITE!", "error");
            //return false;
        } else if (MontoDescuento > 100) {// Segundo chequeo: si el descuento sobrepasa el 100%
            $(this).val("0");
            Utils.showAlert("EL DESCUENTO NO PUEDE SER MAYOR A 100% !", "error");
            //return false;
        }
        _cartManager.updateCart({}); 
    });

    // FUNCION PARA CAMBIAR PRECIOS PRODUCTOS
    window.CargaPreciosProductos = function() {
        var tipo_precio = $('input:radio[name=preciop]:checked').val();
        $('#loading').load("familias_productos?CargarProductos=si&tipo_precio="+tipo_precio);
    };

    // FUNCION PARA CAMBIAR PRECIOS COMBOS
    window.CargaPreciosCombos = function() {
        var tipo_precio = $('input:radio[name=precioc]:checked').val();
        $('#loading').load("familias_productos?CargarCombos=si&tipo_precio="+tipo_precio);
    };

    // ###############################################################
    // # FUNCIONES DE TECLADO CON TECLAS DE FUNCIÓN (F-KEYS) #
    // ###############################################################
    $(document).keydown(function(e) {
        // Inicializa isButtonPagoDisabled de forma segura.
        const isButtonPagoDisabled = $("#buttonpago").prop("disabled");

        switch (e.keyCode) {
            case 13: // ENTER - Agrega detalle de producto
                // Modificado para evitar doble disparo si el foco está en un input o textarea
                if (e.target.id !== 'txtCodigoBarras' && e.target.tagName.toLowerCase() !== 'textarea' && e.target.tagName.toLowerCase() !== 'input') {
                    $('#AgregaProducto').trigger("click");
                    return false;
                }
            break;

            case 27: // ESCAPE
                // Redirige a la URL del formulario o página que deseas
                window.location.href = "panel";
                return false;
            break;

            case 113: // F2 - Muestra modal de pago
                if (!isButtonPagoDisabled) {
                    $("#myModalPago").modal("toggle");
                    $('#myModalPago').one('shown.bs.modal', function() {
                        $('#montopagado').focus();
                    });
                    return false;
                }
            break;

            case 115: // F4 - Limpia valores de formulario (vaciar carrito)
                if (!isButtonPagoDisabled) {
                    $("#myModalPago .close").trigger("click");
                    $('#vaciar').trigger("click");
                    return false;
                }
            break;

            case 118: // F7 - Muestra modal de cliente
                $("#myModalCliente").modal("toggle");
                return false;
            break;

            case 119: // F8 (código 119) - Cobrar/Registrar Venta
                let botonParaActivar = null; // Usaremos esta variable para guardar el botón correcto

                // Intentar encontrar el botón de "Registrar Venta"
                const botonRegistrar = $("#submit_cobrar");
                if (botonRegistrar.length > 0) { // Si #submit_cobrar existe en esta página
                    const estaDeshabilitadoRegistrar = botonRegistrar.prop("disabled");
                    //console.log("DEBUG: Botón #submit_cobrar encontrado. Deshabilitado:", estaDeshabilitadoRegistrar);
                    if (!estaDeshabilitadoRegistrar) { // Si no está deshabilitado
                        botonParaActivar = botonRegistrar;
                    }
                }

                // Si #submit_cobrar no se encontró o estaba deshabilitado, entonces intentamos con #btn-submit
                if (botonParaActivar === null) {
                    const botonActualizar = $("#btn-submit");
                    if (botonActualizar.length > 0) { // Si #btn-submit existe en esta página
                        const estaDeshabilitadoActualizar = botonActualizar.prop("disabled");
                        //console.log("DEBUG: Botón #btn-submit encontrado. Deshabilitado:", estaDeshabilitadoActualizar);
                        if (!estaDeshabilitadoActualizar) { // Si no está deshabilitado
                            botonParaActivar = botonActualizar;
                        }
                    }
                }

                // Finalmente, si encontramos un botón que no está deshabilitado, lo activamos
                if (botonParaActivar !== null) {
                    botonParaActivar.trigger("click"); // Simula un clic en el botón
                    //console.log("DEBUG: Click simulado en el botón con ID:", botonParaActivar.attr('id'));
                    e.preventDefault(); // Evita que el navegador haga algo por defecto con F8
                    return false; // Detiene la propagación del evento
                } else {
                    console.log("DEBUG: F8 presionada, pero no se encontró un botón de cobro/registro activo.");
                }
            break;

            case 120: // F9 - Cierra modal de cobro
                if (!isButtonPagoDisabled) {
                    $("#myModalPago .close").trigger("click");
                    return false;
                }
            break;

            case 121: // F10 - Cierra modal de Apertura de Caja
                $("#myModalArqueo").modal("toggle");
                return false;
            break;
        }
    });

    //################# PROCESO PARA COBRAR VENTA #################
    // FUNCION AGREGAR MEDIO DE PAGO
    window.addRowPago = function() {
        // Asegúrate de que el ID #rowPago exista en tu HTML y contenga el template
        const html = $("#rowPago").html().replace(/\$INDEX/g, $("#muestra_condiciones > .row").length);
        $("#muestra_condiciones").append(html);
    };

    // FUNCION QUITAR MEDIO DE PAGO
    window.rmRowPago = function(el) {
        $(el).closest(".row").remove();
        // Llamada a CalculoDevolucion para recalcular después de quitar
        CalculoDevolucion();
    };

    // FUNCION PARA CALCULAR DEVOLUCION EN VENTA
    window.CalculoDevolucion = function() {
        if ($('input#txtTotal').val() === "0" || $('input#txtTotal').val() === "0") {
            $("#montopagado").val("");
            Utils.showAlert("POR FAVOR AGREGUE DETALLES PARA CONTINUAR CON LA VENTA DE PRODUCTOS!", "error"); // Usando Utils.showAlert
            return false;
        } else {
            let sumaPagos = 0;
            // Suma los valores de todos los inputs dentro de #muestra_condiciones
            $("#muestra_condiciones input").each(function() {
                sumaPagos += parseFloat($(this).val()) || 0; // Asegura que sean números
            });

            const montopagado = sumaPagos;
            const montototal  = parseFloat($("input#txtTotal").val()) || 0; // Asegura que sea número
            // const montodevuelto = $("input#montodevuelto").val(); // No se usa directamente aquí

            // REALIZO EL CALCULO Y MUESTRO LA DEVOLUCION
            const montoVuelto = montopagado - montototal;

            $("#txtPagado").val(montopagado.toFixed(0));
            $("#TextPagado").text(Utils.Separador(montopagado.toFixed(0))); // Usando Utils.Separador
            $("#TextCambio").text((montopagado === 0) ? "0" : Utils.Separador(montoVuelto.toFixed(0))); // Usando Utils.Separador
            $("#montodevuelto").val((montopagado === 0) ? "0" : montoVuelto.toFixed(0));
        }
    };

    // FUNCION PARA MOSTRAR CONDICIONES DE PAGO
    window.CargaCondicionesPagos = function() {
        var tipopago   = $('input:radio[name=tipopago]:checked').val();
        var montototal = $('input#txtTotal').val();

        var sumtotal   = parseFloat(montototal);
        var Sumatoria  = parseFloat(sumtotal.toFixed(0));

        $("#TextImporte").text(Utils.Separador(Sumatoria.toFixed(0))); // Usando Utils.Separador
        $("#TextPagado").text(tipopago == "CREDITO" ? "0" : Utils.Separador(montototal)); // Usando Utils.Separador
        $("#txtPagado").val(tipopago == "CREDITO" ? "0" : montototal);
        $("#TextCambio").text("0");

        var dataString = 'BuscaCondicionesPagos=si&tipopago=' + tipopago + "&txtTotal=" + montototal;

        $.ajax({
            type: "GET",
            url: "condiciones_pagos.php",
            data: dataString,
            success: function(response) {
                $('#muestra_condiciones').empty();
                $('#muestra_condiciones').append('' + response + '').fadeIn("slow");
            }
        });
    };
    //################# PROCESO PARA COBRAR VENTA #################
});
// #endregion EVENT HANDL


// #region LÓGICA DE CARGA DE PRODUCTOS Y COMBOS
/**
 * Función principal para cargar productos o combos mediante AJAX.
 * Esta función es el corazón de la carga dinámica de contenido.
 * @param {string} [familia=''] - ID de la familia/categoría seleccionada.
 * @param {string} [busqueda=''] - Texto de búsqueda.
 * @param {string} [tipoPrecio=''] - Tipo de precio ('1', '2', '3').
 * @param {string} [tipoContenido='productos'] - 'productos' o 'combos'.
 */
window.CargarContenido = function(familia = '', busqueda = '', tipoPrecio = '', tipoContenido = '') {
    const finalTipoPrecio = tipoPrecio || $('input[name="preciop"]:checked').val() || '3'; // Por defecto '3' (Precio Público)

    const currentSearch = $('#busquedaproducto').val();
    const tipoBusqueda  = $("#tipobusqueda").val(); // 'productos' o 'combos' desde el hidden input

    // Si la familia es explícitamente vacía, no aplicamos ningún filtro de familia.
    // Si no, usamos la familia provista o la actual seleccionada.
    let finalFamily = familia; // Por defecto, usa el parámetro 'familia'
    if (familia === '') {
        // Si el parámetro familia es vacío, y el tipo de busqueda es 'productos',
        // entonces queremos usar la familia actualmente seleccionada.
        // Si el tipo de busqueda es 'combos' y el parámetro familia es vacío,
        // queremos ignorar cualquier familia previamente seleccionada.
        if (tipoBusqueda === 'productos') {
            finalFamily = $('.categories.selectedGat').attr('id') || '';
        } else {
            finalFamily = ''; // Para combos, si no se especifica familia, buscamos TODO
        }
    }

    const finalSearch = busqueda === '' ? currentSearch : busqueda;

    // Deshabilitar controles para evitar nuevas peticiones durante la carga
    $('#busquedaproducto').prop('disabled', true);
    $('input[name="preciop"]').prop('disabled', true);
    $('.categories').css('pointer-events', 'none'); // Deshabilita clicks en categorías
    $('button[onClick^="Mostrar"]').prop('disabled', true); // Deshabilita botones de Productos/Combos

    // Preparar los datos y la URL para la petición AJAX
    let dataToSend = {
        busqueda: finalSearch,
        familia: finalFamily, // Ahora finalFamily podría ser realmente '' si se pide
        tipo_precio: finalTipoPrecio
    };

    let urlTarget = "";
    let loadingMessage = "";

    // IMPORTANT: Usar tipoBusqueda del hidden input para determinar si son productos o combos
    if (tipoBusqueda === 'productos') {
        dataToSend.cargar_productos_ventas_pos = "si";
        urlTarget      = "familias_productos.php";
        loadingMessage = "productos";
    } else if (tipoBusqueda === 'combos') {
        dataToSend.cargar_combos_ventas_pos = "si";
        urlTarget      = "familias_productos.php";
        loadingMessage = "combos";
    } else {
        console.error("Tipo de contenido no reconocido: " + tipoBusqueda);
        // Re-habilitar controles si hay un error antes de la petición AJAX
        $('#busquedaproducto').prop('disabled', false);
        $('input[name="preciop"]').prop('disabled', false);
        $('.categories').css('pointer-events', 'auto');
        $('button[onClick^="Mostrar"]').prop('disabled', false);
        return;
    }

    // Mostramos el indicador de carga dentro del contenedor principal de productos
    $('#productList2').html('<div class="loading-indicator text-center p-4"><i class="fa fa-spinner fa-spin fa-3x"></i> Cargando ' + loadingMessage + '...</div>');

    // Realizamos la petición AJAX
    $.ajax({
        type: "GET",
        url: urlTarget,
        data: dataToSend,
        dataType: "html",
        success: function(response) {
            $('#productList2').html(response);
        },
        error: function(jqXHR, textStatus, errorThrown) {
            console.error("Error al cargar " + loadingMessage + ":", textStatus, errorThrown, jqXHR.responseText);
            $('#productList2').html('<div class="alert alert-danger"><center><span class="fa fa-times-circle"></span> Error al cargar ' + loadingMessage + '. Intente de nuevo.</center></div>');
        },
        complete: function() {
            // Re-habilitar controles al finalizar la petición (éxito o error)
            $('#busquedaproducto').prop('disabled', false);
            $('input[name="preciop"]').prop('disabled', false);
            $('.categories').css('pointer-events', 'auto');
            $('button[onClick^="Mostrar"]').prop('disabled', false);
        }
    });
};

// Estas funciones actúan como el punto de entrada para esos eventos.
window.MostrarProductos = function() {
    $('#tipobusqueda').val("productos");
    $('#busquedaproducto').val(''); // Limpiar la búsqueda al cambiar a combos
    // Deseleccionar todos los tabs de familia
    $('.categories').removeClass('selectedGat');
    // Seleccionar el tab de "Home" o el que representa "Todas las categorías"
    $('.categories').first().addClass('selectedGat');
    // Llama a la función principal para cargar productos.
    window.CargarContenido('', '', '', 'productos');
};

window.MostrarCombos = function() {
    $('#tipobusqueda').val("combos");
    $('#busquedaproducto').val(''); // Limpiar la búsqueda al cambiar a combos
    // Deseleccionar todos los tabs de familia
    $('.categories').removeClass('selectedGat');
    // Seleccionar el tab de "Home" o el que representa "Todas las categorías"
    $('.categories').first().addClass('selectedGat');
    // Llama a la función principal para cargar combos.
    window.CargarContenido('', '', '', 'combos');
};
// #endregion LÓGICA DE CARGA DE PRODUCTOS Y COMBOS (UNIFICADA)

// #region DOCUMENT READY (Eventos al cargar la página y listeners de elementos)
$(document).ready(function() {

    // Carga inicial de productos al cargar el módulo POS
    window.CargarContenido('', '', '', 'productos');

    // Función para inicializar todos los listeners de eventos
    function initEventListeners() {
        // Manejar clic en las categorías (tabs de familia)
        $('.categories').on('click', function() {
            $('.categories').removeClass('selectedGat');
            $(this).addClass('selectedGat');
            const familiaId = $(this).attr('id');
            $('#busquedaproducto').val('');
            const tipoContenidoActual = $('#tipobusqueda').val(); // Leer el tipo de búsqueda actual
            window.CargarContenido(familiaId, '', '', tipoContenidoActual);
        });

        // Manejar la búsqueda por texto (con "debounce")
        let typingTimer;
        const doneTypingInterval = 300;

        $('#busquedaproducto').on('keyup', function() {
            clearTimeout(typingTimer);
            const searchText = $(this).val();
            const tipoBusqueda = $("#tipobusqueda").val();
            if (searchText.length >= 2 || searchText === '') {
                typingTimer = setTimeout(function() {
                    window.CargarContenido('', searchText, '', tipoBusqueda);
                }, doneTypingInterval);
            }
        });

        // Manejar el cambio de tipo de precio (radio buttons)
        $('input[name="preciop"]').on('change', function() {
            const tipoPrecio = $(this).val();
            const tipoBusqueda = $("#tipobusqueda").val();
            window.CargarContenido('', '', tipoPrecio, tipoBusqueda);
        });
    }

    // Inicializar los listeners
    initEventListeners();

    // Asegúrate de que el radio button de precio público esté seleccionado al cargar la página
    $('input[name="preciop"][value="3"]').prop('checked', true);

    // Opcional: Recalcular montos si cambian los campos de subtotal, descuento, pagado.
    $('#txtSubtotal, #txtDescuento, #txtPagado').on('input', function() {
        Utils.CalMontos();
    });
});
// #endregion DOCUMENT READY