<?php
require_once('class/class.php');
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="icon" type="image/png" sizes="16x16" href="assets/images/favicon.png">
    <title>Verificador de Precios</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet" integrity="sha384-QWTKZyjpPEjISv5WaRU9OFeRpok6YctnYmDr5pNlyT2bRjXh0JMhjY6hW+ALEwIH" crossorigin="anonymous">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" integrity="sha512-SnH5WK+bZxgPHs44uWIX+LLJAJ9/2PkPKZ5QiAj6Ta86w+fsb2TkcmfRyVX3pBnMFcV7oQPJkl9QevSCWr3W6A==" crossorigin="anonymous" referrerpolicy="no-referrer" />
    <style>
    body {
        background-color: #f0f2f5; /* Un gris más suave */
        font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; /* Fuente moderna */
        display: flex;
        justify-content: center;
        align-items: center;
        min-height: 100vh; /* Usa min-height para asegurar que ocupe toda la altura */
        margin: 0;
        overflow: hidden; /* Evita barras de desplazamiento innecesarias */
    }
    .container-verificador {
        background-color: #ffffff;
        border-radius: 15px;
        box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1); /* Sombra más pronunciada */
        padding: 40px;
        text-align: center;
        max-width: 600px; /* Ancho máximo para el contenedor principal */
        width: 90%; /* Ancho responsivo */
        animation: fadeIn 0.8s ease-out; /* Animación de entrada */
    }
    @keyframes fadeIn {
        from { opacity: 0; transform: translateY(-20px); }
        to { opacity: 1; transform: translateY(0); }
    }

    h1 {
        color: #343a40;
        margin-bottom: 25px;
        font-size: 2.2rem; /* Tamaño de título ajustado */
    }
    #codigo {
        text-align: center;
        font-size: 1.8rem; /* Un poco más pequeño para mejor ajuste */
        padding: 15px 10px;
        border-radius: 8px;
        border: 2px solid #007bff; /* Borde con color primario */
        transition: border-color 0.3s ease;
    }
    #codigo:focus {
        border-color: #0056b3; /* Borde más oscuro al enfocar */
        box-shadow: 0 0 0 0.25rem rgba(0, 123, 255, 0.25); /* Sombra al enfocar */
    }

    #resultado {
        margin-top: 30px;
        min-height: 250px; /* Altura mínima para evitar saltos de contenido */
        display: flex; /* Aseguramos flex para centrar contenido */
        flex-direction: column; /* Centramos en columna por defecto */
        justify-content: center; /* Centrado vertical */
        align-items: center; /* Centrado horizontal */
        position: relative;
    }

    .producto-img {
        max-width: 250px; /* Tamaño un poco más grande */
        max-height: 250px;
        width: auto; /* Asegura que la imagen se adapte al ancho */
        height: auto;
        margin: 15px auto;
        border-radius: 8px;
        box-shadow: 0 5px 15px rgba(0, 0, 0, 0.08);
        object-fit: contain; /* Asegura que la imagen no se recorte */
        transition: opacity 0.5s ease-in-out, transform 0.5s ease-in-out; /* Transiciones suaves */
    }

    .nombre-producto {
        font-size: 2.5rem; /* Tamaño grande para el nombre */
        color: #343a40;
        margin-bottom: 10px;
        min-height: 3rem; /* Altura mínima para el nombre */
        line-height: 1.2;
        transition: opacity 0.5s ease-in-out;
    }

    .precio {
        font-size: 6.5rem; /* Tamaño adecuado para el precio en pantalla completa */
        font-weight: bold;
        color: #28a745; /* Verde para el precio */
        margin-bottom: 0;
        transition: opacity 0.5s ease-in-out;
    }

    .message-text {
        font-size: 2rem;
        color: #6c757d;
    }
    .error-text {
        color: #dc3545; /* Rojo para errores */
    }

    /* Ocultar elementos con transiciones */
    .fade-out {
        opacity: 0 !important;
        transform: scale(0.95);
    }
    .fade-in {
        opacity: 1 !important;
        transform: scale(1);
    }

    /* Media queries para ajustar el tamaño de fuente en móviles */
    @media (max-width: 768px) {
        .container-verificador {
            padding: 25px;
        }
        h1 {
            font-size: 1.8rem;
        }
        #codigo {
            font-size: 1.5rem;
        }
        .nombre-producto {
            font-size: 1.8rem;
        }
        .precio {
            font-size: 4rem; /* Precio más pequeño en móviles */
        }
        .producto-img {
            max-width: 150px;
            max-height: 150px;
        }
    }
    @media (max-width: 480px) {
        .precio {
            font-size: 3.5rem;
        }
    }

    .d-none {
        display: none !important;
    }
    </style>
</head>
<body>
    <div class="container-verificador">
        <h1 id="mainTitle"><i class="fas fa-barcode"></i> Verificador de Precios</h1>
        <input type="hidden" id="codsucursal" value="1">
        <input type="text" id="codigo" class="form-control" placeholder="Escanee el código de barras..." autofocus>

        <div id="resultado" class="mt-4">
            <h2 id="nombreProducto" class="nombre-producto message-text">¡Escanee un producto!</h2>
            <img id="imagenProducto" class="producto-img d-none" alt="Imagen del Producto" onerror="this.onerror=null;this.src='assets/images/no-image.jpg'; this.classList.remove('d-none');">
            <p class="precio d-none" id="precioProducto"></p>
            <!--<div id="loadingMessage" class="loading-message-text d-none">Cargando producto...</div>-->
        </div>
    </div>
    <script src="assets/js/bootstrap.bundle.min.js"></script>
    <script src="assets/js/jquery-3.6.0.min.js"></script>
    <script>
    $(document).ready(function() {
        const $codigoInput        = $('#codigo');
        const $nombreProducto     = $('#nombreProducto');
        const $imagenProducto     = $('#imagenProducto');
        const $precioProducto     = $('#precioProducto');
        const $resultadoContainer = $('#resultado');

        let currentResetTimeout;

        // Función para mostrar el estado de carga
        function showLoading() {
        clearTimeout(currentResetTimeout);

            // Oculta todos los elementos de resultado para la transición
            $nombreProducto.addClass('fade-out');
            $imagenProducto.addClass('fade-out');
            $precioProducto.addClass('fade-out');

            setTimeout(() => {
                $nombreProducto.text('Buscando...');
                $nombreProducto.removeClass('fade-in error-text').addClass('message-text');
                $imagenProducto.addClass('d-none').attr('src', '');
                $precioProducto.addClass('d-none').text('');
            }, 300);
        }

        // Función para mostrar el resultado (éxito o error)
        function showResult(nombre, precio, imagenSrc, isError = false) {
            clearTimeout(currentResetTimeout);

            $nombreProducto.removeClass('fade-in message-text error-text').addClass('fade-out');
            $imagenProducto.addClass('fade-out');
            $precioProducto.addClass('fade-out');

            setTimeout(() => {
                $nombreProducto.text(nombre);
                $precioProducto.text(precio);

                if (isError) {
                    $nombreProducto.addClass('error-text');
                    $precioProducto.addClass('d-none').text('');
                    $imagenProducto.addClass('d-none').attr('src', '');
                } else {
                    $nombreProducto.removeClass('error-text message-text');
                    $precioProducto.removeClass('d-none').addClass('fade-in');

                    if (imagenSrc) {
                        const img   = new Image();
                        img.onload  = function() {
                            $imagenProducto.attr('src', imagenSrc).removeClass('d-none').addClass('fade-in');
                        };
                        img.onerror = function() {
                            $imagenProducto.attr('src', 'assets/images/no-image.png').removeClass('d-none').addClass('fade-in');
                            console.warn("No se pudo cargar la imagen:", imagenSrc, ". Usando fallback.");
                        };
                        img.src = imagenSrc;
                    } else {
                        $imagenProducto.attr('src', 'assets/images/no-image.png').removeClass('d-none').addClass('fade-in');
                    }
                }
                $nombreProducto.removeClass('fade-out').addClass('fade-in');

                currentResetTimeout = setTimeout(() => {
                    resetVerifier();
                }, 8000);
            }, 300);
        }

        // Función para resetear el verificador
        function resetVerifier() {
            clearTimeout(currentResetTimeout);

            $nombreProducto.addClass('fade-out');
            $imagenProducto.addClass('fade-out');
            $precioProducto.addClass('fade-out');

            setTimeout(() => {
                $nombreProducto.text('¡Escanee un producto!');
                $nombreProducto.removeClass('fade-out fade-in error-text').addClass('message-text');
                $imagenProducto.addClass('d-none').attr('src', '');
                $precioProducto.addClass('d-none').text('');
                $codigoInput.val('').focus();
            }, 300);
        }

        // Evento al escanear/introducir el código
        $codigoInput.on("change", function () {
            let codsucursal = $('#codsucursal').val();
            let codeBar = $(this).val().trim();

            if (codeBar === "") {
                resetVerifier();
                return;
            }
            showLoading(); // Muestra el mensaje de carga
            $.ajax({
                url: "class/busqueda_autocompleto.php?Busqueda_Producto_Barcode_Verificador=si",
                data: { barcodev: codeBar, codsucursal: codsucursal },
                type: 'POST',
                dataType: 'json',
                success: function (json) {
                    try {
                        let producto = json[0];

                        if (!producto || typeof producto !== 'object' || Object.keys(producto).length === 0) {
                            showResult("Producto no encontrado", "", null, true);
                            return;
                        }
                        if (!producto.simbolo || !producto.producto || producto.precioxpublico === undefined || producto.precioxpublico === null) {
                            console.warn("Producto encontrado con datos incompletos:", producto);
                            showResult("Datos incompletos del producto", "N/A", null, true);
                            return;
                        }
                        let moneda          = producto.simbolo;
                        let nombre          = producto.producto;
                        let precioFormatted = parseFloat(producto.precioxpublico).toLocaleString('es-CL', {
                            minimumFractionDigits: 0,
                            maximumFractionDigits: 0
                        });
                        let precioDisplay = moneda + ' ' + precioFormatted;
                        let imagenSrc     = `fotos/productos/${producto.codsucursal}/${producto.codproducto}.jpg`;
                        showResult(nombre, precioDisplay, imagenSrc);

                    } catch (e) {
                        console.error("Error al procesar el JSON del producto:", e, "JSON recibido:", json);
                        showResult("Error interno al leer datos", "", null, true);
                    }
                },
                error: function (xhr, status, error) {
                    console.error("Error en la consulta AJAX:", status, error, "Respuesta:", xhr.responseText);
                    let errorMessage = "Error de conexión al servidor.";
                    if (xhr.status === 404) {
                        errorMessage = "Recurso no encontrado (404).";
                    } else if (xhr.status === 500) {
                        errorMessage = "Error interno del servidor (500).";
                    } else {
                         errorMessage += ` (Estado: ${xhr.status})`;
                    }
                    showResult(errorMessage, "", null, true);
                },
                complete: function() {
                    $codigoInput.val('');
                }
            });
        });
        // Inicializar el verificador
        resetVerifier();
    });
    </script>
</body>
</html>